/******************************************************************************\
**
**     COMPANY:	LDW Software
**
**      MODULE:	BehaviorEditor
**
** DESCRIPTION:	Editor for creating behaviors
**
**      AUTHOR:	Adrian Francis
**
\******************************************************************************/

#ifndef BEHAVIOR_EDITOR_H
#define BEHAVIOR_EDITOR_H

/******************************************************************************\
** LIBRARY INCLUDES
\******************************************************************************/
#include <stdio.h>
#include "ldwTypes.h"
#include "EditorInterface.h"
#include "../Villagers/Villager.h"
#include "../World/ContentMap.h"
#include "../World/Waypoint.h"

/******************************************************************************\
** CONSTANT DEFINITIONS
\******************************************************************************/
const int BehaviorEditor_c_iMaxControlPoints = 32;
const int BehaviorEditor_c_iMaxPlansAtControlPoint = 8;
const int BehaviorEditor_c_iMaxLinks = 4;

/******************************************************************************\
** CLASS DEFINITIONS
\******************************************************************************/
class CBehaviorEditor : public IEditor
{
private:

	//class enumerations
	enum EPlan
	{	
		ePlan_None = 0,

//		ePlan_AdvanceState,
		ePlan_Bend,
		ePlan_Carry,
		ePlan_Climb,
		ePlan_Dance,
		ePlan_Dive,
		ePlan_Drop,
		ePlan_Embrace,
		ePlan_FaceDirection,
		ePlan_Go,
		ePlan_Heal,
		ePlan_Hide,
		ePlan_Jump,
		ePlan_Learn,
		ePlan_PlaySound,
		ePlan_Say,
		ePlan_ShakeHead,
		ePlan_StopSound,
		ePlan_Swim,
//		ePlan_TakeCorpse,
		ePlan_Wander,
		ePlan_Wait,
		ePlan_Work,

		ePlan_NumberOf
	};

	enum EControlPoint
	{
		eControlPoint_Waypoint,
		eControlPoint_WaypointOffset,
		eControlPoint_Object,
		eControlPoint_Absolute,
	};

	//class structures
	struct SPlan
	{
		EPlan					m_eType;
		EBodyPosition			m_eBodyPosition;
		ECarrying				m_eCarrying;
		EDirection				m_eDirection;
		EHeadDirection			m_eHeadDirection;
		ESkill					m_eSkill;
		ESound					m_eSound;
		int						m_iTime;
	};

	struct SControlPoint
	{
		bool					m_bIsActive;

		EControlPoint			m_eType;
		EWaypoint				m_eWaypoint;
		CContentMap::EObject	m_eObject;
		ldwPoint				m_sOffset;

		bool					m_bIsFuzzy;
		ldwPoint				m_sFuzz;

		SPlan					m_sPlan[BehaviorEditor_c_iMaxPlansAtControlPoint];
		int						m_iNumPlans;

		int						m_iLink[BehaviorEditor_c_iMaxLinks];
		int						m_iNumLinks;
	};

	//class data
	EPlan						m_ePlanSelection;
	EBodyPosition				m_eBodyPositionSelection;
	ECarrying					m_eCarryingSelection;
	EDirection					m_eDirectionSelection;
	EHeadDirection				m_eHeadDirectionSelection;
	ESkill						m_eSkillSelection;
	ESound						m_eSoundSelection;
	int							m_iTimeSelection;

	bool						m_bIsBodyPositionSelectionEnabled;
	bool						m_bIsCarryingSelectionEnabled;
	bool						m_bIsDirectionSelectionEnabled;
	bool						m_bIsHeadDirectionSelectionEnabled;
	bool						m_bIsSkillSelectionEnabled;
	bool						m_bIsSoundSelectionEnabled;
	bool						m_bIsTimeSelectionEnabled;

	EWaypoint					m_eWaypointSelection;
	CContentMap::EObject		m_eObjectSelection;
	int							m_iControlPointSelection;

	bool						m_bIsMouseDown;

	bool						m_bIsEditingEnabled;
	bool						m_bIsReverseEnabled;
	bool						m_bIsInsertEnabled;
	bool						m_bUseAbsoluteCoords;

	SControlPoint				m_sControlPoint[BehaviorEditor_c_iMaxControlPoints];

	int							m_iIntersectedLineStart;
	int							m_iIntersectedLineEnd;
	ldwPoint					m_sIntersectionPoint;

	int							m_iPlanSelection;

	int							m_iNumTabs;

	//class methods
	//setup
	public:		CBehaviorEditor(void);
	public:		void Reset();

	// process
	private:	void GenerateBehaviorCode(const char *psName);
	private:	void GenerateCodeAtControlPoint(FILE *pFile, const int iControlPoint);
	private:	const char *GenerateOffsetPoint(const SControlPoint &sControlPoint);
	private:	void AddLine(FILE *pFile, const char *psLine, ...);
	private:	void AddControlPointPlans(FILE *pFile, const SControlPoint &sControlPoint);

	private:	const int AddControlPoint(const int iControlPoint);
	private:	const int AddControlPoint(const int iControlPoint, const EWaypoint eWaypoint);
	private:	const int AddControlPoint(const int iControlPoint, const CContentMap::EObject eObject);
	private:	const int AddControlPoint(const int iControlPoint, const ldwPoint sOffset);
	private:	const int InsertControlPoint(const int iControlPoint1, const int iControlPoint2, const ldwPoint sOffset);
	private:	void DeleteControlPoint(const int iControlPoint);
	private:	void AddLink(SControlPoint &sControlPoint, const int iControlPoint);
	private:	void DeleteLink(SControlPoint &sControlPoint, const int iControlPoint);
	private:	const ldwPoint GetControlPoint(const int iControlPoint) const;
	private:	const int GetNearbyControlPoint(const ldwPoint sPos) const;
	private:	void SetControlPoint(const int iControlPoint, const ldwPoint sPos);
	private:	const bool SelectControlPointAtWaypoint(const EWaypoint eWaypoint);

	private:	void AddPlanToControlPoint(const int iControlPoint);
	private:	void RemovePlanFromControlPoint(const int iControlPoint, const int iPlan);
	private:	void MovePlanUp(const int iControlPoint, int &iPlan);
	private:	void MovePlanDown(const int iControlPoint, int &iPlan);
	private:	void UpdatePlanSelections();
	private:	void ToggleSelectionsForPlan(const EPlan ePlan);

	public:		void Draw();
	private:	void DrawControlPoint(const int iControlPoint) const;
	public:		const bool HandleKeyCharacter(const char cKey);
	public:		const bool HandleKeyDown(const int iKeyCode);
	public:		const bool HandleKeyUp(const int iKeyCode);
	public:		const bool HandleMouseDown(const ldwPoint sMousePos);
	public:		const bool HandleMouseUp(const ldwPoint sMousePos);
	public:		const bool HandleMouseMove(const ldwPoint sMousePos);

	public:		void FindNearestLine(const int iControlPoint, const ldwPoint sPos, int &iShortestDist, int &iNearestLineStart, int &iNearestLineEnd);

	// access
//	private:	const char *GetModeName(const EMode eMode) const;
	private:	const char *GetPlanName(const EPlan ePlan) const;
	private:	const char *GetSkillName(const ESkill eSkill) const;
	private:	const char *GetSkillEnumName(const ESkill eSkill) const;
	private:	const char *GetBodyPositionName(const EBodyPosition eBodyPosition) const;
	private:	const char *GetBodyPositionEnumName(const EBodyPosition eBodyPosition) const;
	private:	const char *GetCarryingName(const ECarrying eCarrying) const;
	private:	const char *GetCarryingEnumName(const ECarrying eCarrying) const;
	private:	const char *GetDirectionName(const EDirection eDirection) const;
	private:	const char *GetDirectionEnumName(const EDirection eDirection) const;
	private:	const char *GetHeadDirectionName(const EHeadDirection eHeadDirection) const;
	private:	const char *GetHeadDirectionEnumName(const EHeadDirection eHeadDirection) const;
	private:	const char *GetSoundName(const ESound eSound) const;
	private:	const char *GetSoundEnumName(const ESound eSound) const;
};

/******************************************************************************\
** GLOBAL DECLARATIONS
\******************************************************************************/
extern CBehaviorEditor		BehaviorEditor;

#endif

/******************************************************************************\
** END OF FILE
\******************************************************************************/